//ditty.h
//The Ditty Player v0.5 (beta)
//(c) Copyright 2001 John Sensebe
//This code is free to use for non-commercial purposes
//See license.txt for details

#ifndef _DITTY_H_
#define _DITTY_H_

//standard types
#define u8 unsigned char
#define u16 unsigned short
#define u32 unsigned int

//The structure of the Ditty Header
typedef struct {
	char DITY[4];		//The literal characters 'DITY'
	u32 length;		//The length, in bytes, of the Ditty data (not including the header)
	u8 songs[32];		//The ending index in the playorder for each song
	u16 segments[256];	//The starting index in the Ditty data for each segment (after the first)
	u8 playorder[256];	//The segment numbers to play for each song
	u8 reserved[216];	//Reserved for future use
} DittyHeader;

//There are 8 frequencies between notes, and 12 half-steps per octave.
#define OCTAVE	(12*8)

//The Ditty Player has a range of 6 octaves.
#define NOTE_RANGE	(OCTAVE*6)


//Ditty Player command tokens
#define CMD_MISC	0x00
#define CMD_NOTE1	0x01
#define CMD_NOTE2	0x02
#define CMD_NOTE3	0x03
#define CMD_NOTE4	0x04
#define CMD_SWEEP1	0x05
#define CMD_ENVELOPE1	0x06
#define CMD_ENVELOPE2	0x07
#define CMD_ENVELOPE3	0x08
#define CMD_ENVELOPE4	0x09
#define CMD_DUTYCYCLE1	0x0A
#define CMD_DUTYCYCLE2	0x0B
#define CMD_OUTPUTLVL3	0x0C
#define CMD_NOISEDATA4	0x0D
#define CMD_SOUNDLEN1	0x0E
#define CMD_SOUNDLEN2	0x0F
#define CMD_SOUNDLEN3	0x10
#define CMD_SOUNDLEN4	0x11
#define CMD_CONTINUOUS1	0x12
#define CMD_CONTINUOUS2	0x13
#define CMD_CONTINUOUS3	0x14
#define CMD_CONTINUOUS4	0x15
#define CMD_TEMPO	0x16
#define CMD_USERFLAG	0x17
#define CMD_WAVEDATA_0	0x18
#define CMD_WAVEDATA_1	0x19
#define CMD_WAVEDATA_2	0x1A
#define CMD_WAVEDATA_3	0x1B
#define CMD_WAVEDATA_4	0x1C
#define CMD_WAVEDATA_5	0x1D
#define CMD_WAVEDATA_6	0x1E
#define CMD_WAVEDATA_7	0x1F
#define CMD_WAVEDATA_8	0x20
#define CMD_WAVEDATA_9	0x21
#define CMD_WAVEDATA_A	0x22
#define CMD_WAVEDATA_B	0x23
#define CMD_WAVEDATA_C	0x24
#define CMD_WAVEDATA_D	0x25
#define CMD_WAVEDATA_E	0x26
#define CMD_WAVEDATA_F	0x27
#define CMD_VOLUME	0x28
#define CMD_STEREO1	0x29
#define CMD_STEREO2	0x2A
#define CMD_STEREO3	0x2B
#define CMD_STEREO4	0x2C
#define CMD_DEFINE	0x2D
#define CMD_ENDDEF	0x2E
#define CMD_CALL	0x2F
#define CMD_JIFFYSTEP	0x30
#define CMD_WAVESWEEP3	0x31

#define CMD_DETUNE1	0x38
#define CMD_DETUNE2	0x39
#define CMD_DETUNE3	0x3A

#define CMD_TRANSPOSE1	0x3C
#define CMD_TRANSPOSE2	0x3D
#define CMD_TRANSPOSE3	0x3E
#define CMD_CLOCKSWEEP4	0x3F
#define CMD_ARPEGGIO1	0x40
#define CMD_ARPEGGIO2	0x41
#define CMD_ARPEGGIO3	0x42

#define CMD_SLUR1	0x44
#define CMD_SLUR2	0x45
#define CMD_SLUR3	0x46
#define CMD_RATIOSWEEP4	0x47
#define CMD_PORTAMENTO1	0x48
#define CMD_PORTAMENTO2	0x49
#define CMD_PORTAMENTO3	0x4A

#define CMD_VIBRATO1	0x4C
#define CMD_VIBRATO2	0x4D
#define CMD_VIBRATO3	0x4E
#define CMD_STEPSWITCH4	0x4F
#define CMD_NOTEDELAY1	0x50
#define CMD_NOTEDELAY2	0x51
#define CMD_NOTEDELAY3	0x52
#define CMD_NOTEDELAY4	0x53
#define CMD_NOTECUT1	0x54
#define CMD_NOTECUT2	0x55
#define CMD_NOTECUT3	0x56
#define CMD_NOTECUT4	0x57
#define CMD_NOTERETRIG1	0x58
#define CMD_NOTERETRIG2	0x59
#define CMD_NOTERETRIG3	0x5A
#define CMD_NOTERETRIG4	0x5B
#define CMD_PULSESWEEP1	0x5C
#define CMD_PULSESWEEP2	0x5D
#define CMD_VOLSWEEP3	0x5E
#define CMD_NEXTSEG	0x5F
#define CMD_VIBRDELAY1	0x60
#define CMD_VIBRDELAY2	0x61
#define CMD_VIBRDELAY3	0x62

//used to limit size of arrays
#define DITTY_COMMANDS	0x63

//Flags used by DittyStereo()
#define STEREO_LEFT_1	0x10
#define STEREO_LEFT_2	0x20
#define STEREO_LEFT_3	0x40
#define STEREO_LEFT_4	0x80
#define STEREO_RIGHT_1	0x01
#define STEREO_RIGHT_2	0x02
#define STEREO_RIGHT_3	0x04
#define STEREO_RIGHT_4	0x08

//Callback definition for function called when each note is struck.
typedef void DittyNoteCallback(const u32 Note,const u32 Voice);

//Callback definition for function called when a User Flag command is executed.
typedef void DittyFlagCallback(const u32 Flag);

//Callback definition for function called when the end of a song is reached.
typedef void DittyEndCallback(void);

//Note GBA frequency values
extern const unsigned short NoteVal[NOTE_RANGE];

//DittySet() must be called first. It sets the internal pointers to the song data.
//If the Ditty is from a save RAM image, sd==sh+1024.
extern void DittySet(const DittyHeader* const sh,const u8* const sd);

//DittyPlay() initializes Ditty song playback.
extern void DittyPlay(const int song);

//DittyPlaySegment initializes Ditty segment playback.
extern void DittyPlaySegment(const int segment);

//DittyPlayJiffy() plays one "jiffy" of a song. It should be called about every 1/60 seconds
//You could simply call it in you game loop, or as part of an interrupt routine.
extern void DittyPlayJiffy(void);

//DittyStop() stops Ditty playback.
extern DittyEndCallback DittyStop;

//DittyLoop() restarts the Ditty song from its beginning.
//It can be used as a callback to loop the song once it's finished.
extern DittyEndCallback DittyLoop;

//DittyGetTick() gets the current tick number in the currently playing segment.
extern int DittyGetTick(void);

//DittyGetSegment() gets the currently playing segment number.
extern int DittyGetSegment(void);

//DittyGetPlayPos() gets the postion in the segment playlist of the currently playing segment.
extern int DittyGetPlayPos(void);

//DittySetNoteCallback() sets the function to be called when a note is struck.
extern void DittySetNoteCallback(DittyNoteCallback *D);

//DittySetFlagCallback() sets the function to be called when a User Flag command is executed.
extern void DittySetFlagCallback(DittyFlagCallback *D);

//DittySetEndCallback() sets the function to be called when a Ditty song finishes.
//If no callback is set, the Player calls DittyStop().
extern void DittySetEndCallback(DittyEndCallback *D);

//DittySetVolume() sets the software volume for both left and right channels.
//The actual volume is a combination of the the volume set by this call, and the Volume command in the Ditty.
//The values go from 0 to 7.
extern void DittySetVolume(const int Left, const int Right);

//DittyStereo() allows the user to activate or deactivate each voice playing on each channel.
//For a voice to be active on a channel, it must be activated by in these flags and by the Stereo command in the song.
extern void DittySetStereo(const int Flags);

//DittyPlaying() returns 1 if the Player is playing, 0 if not. 
extern int DittyPlaying(void);

#endif
