//ditty.c
//The Ditty Player v0.5 (beta)
//(c) Copyright 2001 John Sensebe
//This code is free to use for non-commercial purposes
//See license.txt for details

#include "ditty.h"

//register access macros
#define REG8(x) (*(volatile u8*)(x))
#define REG16(x) (*(volatile u16*)(x))
#define REG32(x) (*(volatile u32*)(x))

//I/O register macros
#define BASE_IO 0x4000000

#define REG_SG10 (BASE_IO+0x60)
#define REG_SG10_L REG_SG10
#define REG_NR10 REG_SG10_L
#define REG_SG10_H (REG_SG10+2)
#define REG_NR11 REG_SG10_H
#define REG_NR12 (REG_SG10_H+1)
#define REG_SG11 (BASE_IO+0x64)
#define REG_NR13 REG_SG11
#define REG_NR14 (REG_SG11+1)
#define REG_SG20 (BASE_IO+0x68)
#define REG_NR21 REG_SG20
#define REG_NR22 (REG_SG20+1)
#define REG_SG21 (BASE_IO+0x6C)
#define REG_NR23 REG_SG21
#define REG_NR24 (REG_SG21+1)
#define REG_SG30 (BASE_IO+0x70)
#define REG_SG30_L REG_SG30
#define REG_NR30 REG_SG30_L
#define REG_SG30_H (REG_SG30+2)
#define REG_NR31 REG_SG30_H
#define REG_NR32 (REG_SG30_H+1)
#define REG_SG31 (BASE_IO+0x74)
#define REG_NR33 REG_SG31
#define REG_NR34 (REG_SG31+1)
#define REG_SG40 (BASE_IO+0x78)
#define REG_NR41 REG_SG40
#define REG_NR42 (REG_SG40+1)
#define REG_SG41 (BASE_IO+0x7C)
#define REG_NR43 REG_SG41
#define REG_NR44 (REG_SG41+1)
#define REG_SGCNT0 (BASE_IO+0x80)
#define REG_SGCNT0_L REG_SGCNT0
#define REG_NR50 REG_SGCNT0_L
#define REG_NR51 (REG_SGCNT0_L+1)
#define REG_SGCNT0_H (REG_SGCNT0+2)
#define REG_SGCNT1 (BASE_IO+0x84)
#define REG_NR52 REG_SGCNT1
#define REG_SGCNT1_L REG_SGCNT1
#define REG_SGCNT1_H (REG_SGCNT1+2)
#define REG_SGBIAS (BASE_IO+0x88)

#define REG_SGWR0 (BASE_IO+0x90)
#define REG_SGWR0_L REG_SGWR0
#define REG_SGWR0_H (REG_SGWR0+2)
#define REG_SGWR1 (BASE_IO+0x94)
#define REG_SGWR1_L REG_SGWR1
#define REG_SGWR1_H (REG_SGWR1+2)
#define REG_SGWR2 (BASE_IO+0x98)
#define REG_SGWR2_L REG_SGWR2
#define REG_SGWR2_H (REG_SGWR2+2)
#define REG_SGWR3 (BASE_IO+0x9C)
#define REG_SGWR3_L REG_SGWR3
#define REG_SGWR3_H (REG_SGWR3+2)

//Define inline for ARM SDT
#ifndef __GNUC__
#define inline __inline
#endif

//GBA frequency values
const unsigned short NoteVal[NOTE_RANGE]={
    44,  58,  73,  87, 101, 115, 129, 143,  // C-3
   157, 170, 184, 197, 210, 224, 237, 250,  // C#3
   263, 276, 288, 301, 313, 326, 338, 351,  // D-3
   363, 375, 387, 399, 411, 423, 434, 446,  // D#3
   457, 469, 480, 492, 503, 514, 525, 536,  // E-3
   547, 558, 568, 579, 589, 600, 610, 621,  // F-3
   631, 641, 651, 661, 671, 681, 691, 701,  // F#3
   711, 720, 730, 739, 749, 758, 767, 776,  // G-3
   786, 795, 804, 813, 822, 830, 839, 848,  // G#3
   856, 865, 874, 882, 890, 899, 907, 915,  // A-3
   923, 931, 939, 947, 955, 963, 971, 979,  // A#3
   986, 994,1002,1009,1017,1024,1031,1039,  // B-3
  1046,1053,1060,1067,1075,1082,1088,1095,  // C-4
  1102,1109,1116,1123,1129,1136,1142,1149,  // C#4
  1155,1162,1168,1174,1181,1187,1193,1199,  // D-4
  1205,1211,1218,1223,1229,1235,1241,1247,  // D#4
  1253,1258,1264,1270,1275,1281,1286,1292,  // E-4
  1297,1303,1308,1313,1319,1324,1329,1334,  // F-4
  1339,1345,1350,1355,1360,1365,1370,1374,  // F#4
  1379,1384,1389,1394,1398,1403,1408,1412,  // G-4
  1417,1421,1426,1430,1435,1439,1444,1448,  // G#4
  1452,1457,1461,1465,1469,1473,1477,1482,  // A-4
  1486,1490,1494,1498,1502,1506,1509,1513,  // A#4
  1517,1521,1525,1529,1532,1536,1540,1543,  // B-4
  1547,1551,1554,1558,1561,1565,1568,1572,  // C-5
  1575,1579,1582,1585,1589,1592,1595,1598,  // C#5
  1602,1605,1608,1611,1614,1617,1621,1624,  // D-5
  1627,1630,1633,1636,1639,1642,1645,1647,  // D#5
  1650,1653,1656,1659,1662,1664,1667,1670,  // E-5
  1673,1675,1678,1681,1683,1686,1689,1691,  // F-5
  1694,1696,1699,1701,1704,1706,1709,1711,  // F#5
  1714,1716,1718,1721,1723,1725,1728,1730,  // G-5
  1732,1735,1737,1739,1741,1744,1746,1748,  // G#5
  1750,1752,1754,1756,1759,1761,1763,1765,  // A-5
  1767,1769,1771,1773,1775,1777,1779,1781,  // A#5
  1783,1785,1786,1788,1790,1792,1794,1796,  // B-5
  1798,1799,1801,1803,1805,1806,1808,1810,  // C-6
  1812,1813,1815,1817,1818,1820,1822,1823,  // C#6
  1825,1826,1828,1830,1831,1833,1834,1836,  // D-6
  1837,1839,1840,1842,1843,1845,1846,1848,  // D#6
  1849,1851,1852,1853,1855,1856,1858,1859,  // E-6
  1860,1862,1863,1864,1866,1867,1868,1870,  // F-6
  1871,1872,1873,1875,1876,1877,1878,1880,  // F#6
  1881,1882,1883,1884,1886,1887,1888,1889,  // G-6
  1890,1891,1892,1894,1895,1896,1897,1898,  // G#6
  1899,1900,1901,1902,1903,1904,1905,1906,  // A-6
  1907,1908,1909,1910,1911,1912,1913,1914,  // A#6
  1915,1916,1917,1918,1919,1920,1921,1922,  // B-6
  1923,1924,1925,1925,1926,1927,1928,1929,  // C-7
  1930,1931,1931,1932,1933,1934,1935,1936,  // C#7
  1936,1937,1938,1939,1940,1940,1941,1942,  // D-7
  1943,1943,1944,1945,1946,1946,1947,1948,  // D#7
  1949,1949,1950,1951,1951,1952,1953,1953,  // E-7
  1954,1955,1956,1956,1957,1957,1958,1959,  // F-7
  1959,1960,1961,1961,1962,1963,1963,1964,  // F#7
  1964,1965,1966,1966,1967,1967,1968,1969,  // G-7
  1969,1970,1970,1971,1971,1972,1972,1973,  // G#7
  1974,1974,1975,1975,1976,1976,1977,1977,  // A-7
  1978,1978,1979,1979,1980,1980,1981,1981,  // A#7
  1982,1982,1983,1983,1984,1984,1984,1985,  // B-7
  1985,1986,1986,1987,1987,1988,1988,1988,  // C-8
  1989,1989,1990,1990,1991,1991,1991,1992,  // C#8
  1992,1993,1993,1993,1994,1994,1995,1995,  // D-8
  1995,1996,1996,1996,1997,1997,1998,1998,  // D#8
  1998,1999,1999,1999,2000,2000,2000,2001,  // E-8
  2001,2001,2002,2002,2002,2003,2003,2003,  // F-8
  2004,2004,2004,2005,2005,2005,2006,2006,  // F#8
  2006,2007,2007,2007,2007,2008,2008,2008,  // G-8
  2009,2009,2009,2009,2010,2010,2010,2010,  // G#8
  2011,2011,2011,2012,2012,2012,2012,2013,  // A-8
  2013,2013,2013,2014,2014,2014,2014,2015,  // A#8
  2015,2015,2015,2016,2016,2016,2016,2016    // B-8
};

//default state (used when the player starts)
static const u8 dittydefaults[DITTY_COMMANDS]={
  0x00,  // CMD_MISC      0x00
  0xFF,  // CMD_NOTE1    0x01
  0xFF,  // CMD_NOTE2    0x02
  0xFF,  // CMD_NOTE3    0x03
  0xFF,  // CMD_NOTE4    0x04
  0x08,  // CMD_SWEEP1    0x05
  0xF2,  // CMD_ENVELOPE1  0x06
  0xF2,  // CMD_ENVELOPE2  0x07
  0xF2,  // CMD_ENVELOPE3  0x08
  0xF1,  // CMD_ENVELOPE4  0x09
  0x02,  // CMD_DUTYCYCLE1  0x0A
  0x02,  // CMD_DUTYCYCLE2  0x0B
  0x01,  // CMD_OUTPUTLVL3  0x0C
  0x40,  // CMD_NOISEDATA4  0x0D
  0x00,  // CMD_SOUNDLEN1  0x0E
  0x00,  // CMD_SOUNDLEN2  0x0F
  0x00,  // CMD_SOUNDLEN3  0x10
  0x00,  // CMD_SOUNDLEN4  0x11
  0x01,  // CMD_CONTINUOUS1  0x12
  0x01,  // CMD_CONTINUOUS2  0x13
  0x01,  // CMD_CONTINUOUS3  0x14
  0x01,  // CMD_CONTINUOUS4  0x15
  0x06,  // CMD_TEMPO    0x16
  0x00,  // CMD_USERFLAG    0x17
  0x0F,  // CMD_WAVEDATA_0  0x18
  0xFF,  // CMD_WAVEDATA_1  0x19
  0xFF,  // CMD_WAVEDATA_2  0x1A
  0xFF,  // CMD_WAVEDATA_3  0x1B
  0xF0,  // CMD_WAVEDATA_4  0x1C
  0x00,  // CMD_WAVEDATA_5  0x1D
  0x00,  // CMD_WAVEDATA_6  0x1E
  0x00,  // CMD_WAVEDATA_7  0x1F
  0x0F,  // CMD_WAVEDATA_8  0x20
  0xFF,  // CMD_WAVEDATA_9  0x21
  0xFF,  // CMD_WAVEDATA_A  0x22
  0xFF,  // CMD_WAVEDATA_B  0x23
  0xF0,  // CMD_WAVEDATA_C  0x24
  0x00,  // CMD_WAVEDATA_D  0x25
  0x00,  // CMD_WAVEDATA_E  0x26
  0x00,  // CMD_WAVEDATA_F  0x27
  0x3F,  // CMD_VOLUME    0x28
  0x03,  // CMD_STEREO1    0x29
  0x03,  // CMD_STEREO2    0x2A
  0x03,  // CMD_STEREO3    0x2B
  0x03,  // CMD_STEREO4    0x2C
  0x00,  // CMD_DEFINE    0x2D
  0x00,  // CMD_ENDDEF    0x2E
  0x00,  // CMD_CALL      0x2F
  0xFF,  // CMD_JIFFYSTEP  0x30
  0x00,  // CMD_WAVESWEEP3  0x31
  0x00,  //          0x32
  0x00,  //          0x33
  0x00,  //          0x34
  0x00,  //          0x35
  0x00,  //          0x36
  0x00,  //          0x37
  0x00,  // CMD_DETUNE1    0x38
  0x00,  // CMD_DETUNE2    0x39
  0x00,  // CMD_DETUNE3    0x3A
  0x00,  //          0x3B
  0x00,  // CMD_TRANSPOSE1  0x3C
  0x00,  // CMD_TRANSPOSE2  0x3D
  0x00,  // CMD_TRANSPOSE3  0x3E
  0x00,  // CMD_CLOCKSWEEP4  0x3F
  0x00,  // CMD_ARPEGGIO1  0x40
  0x00,  // CMD_ARPEGGIO2  0x41
  0x00,  // CMD_ARPEGGIO3  0x42
  0x00,  //          0x43
  0xFF,  // CMD_SLUR1    0x44
  0xFF,  // CMD_SLUR2    0x45
  0xFF,  // CMD_SLUR3    0x46
  0x00,  // CMD_RATIOSWEEP4  0x47
  0x00,  // CMD_PORTAMENTO1  0x48
  0x00,  // CMD_PORTAMENTO2  0x49
  0x00,  // CMD_PORTAMENTO3  0x4A
  0x00,  //          0x4B
  0x00,  // CMD_VIBRATO1    0x4C
  0x00,  // CMD_VIBRATO2    0x4D
  0x00,  // CMD_VIBRATO3    0x4E
  0x00,  // CMD_STEPSWITCH4  0x4F
  0x00,  // CMD_NOTEDELAY1  0x50
  0x00,  // CMD_NOTEDELAY2  0x51
  0x00,  // CMD_NOTEDELAY3  0x52
  0x00,  // CMD_NOTEDELAY4  0x53
  0x00,  // CMD_NOTECUT1    0x54
  0x00,  // CMD_NOTECUT2    0x55
  0x00,  // CMD_NOTECUT3    0x56
  0x00,  // CMD_NOTECUT4    0x57
  0x00,  // CMD_NOTERETRIG1  0x58
  0x00,  // CMD_NOTERETRIG2  0x59
  0x00,  // CMD_NOTERETRIG3  0x5A
  0x00,  // CMD_NOTERETRIG4  0x5B
  0x00,  // CMD_PULSESWEEP1  0x5C
  0x00,  // CMD_PULSESWEEP2  0x5D
  0x00,  // CMD_VOLSWEEP3  0x5E
  0x00,  // CMD_NEXTSEG    0x5F
  0x00,  // CMD_VIBRDELAY1  0x60
  0x00,  // CMD_VIBRDELAY2  0x61
  0x00  // CMD_VIBRDELAY3  0x62
};

static int currentsong;    //Current song (Duh!)
static const u8 *currentcmd;  //Pointer to current command
static int currentplaypos;  //Position in playorder
static int currenttick;
static int currentjiffy;

static const u8 *segend;  //pointer past end of current segment

static const DittyHeader* songheader;  //pointer to song header
static const u8* songdata;        //pointer to song data (usually song header + 0x400)

static u8 dittystate[DITTY_COMMANDS];  //current song state
static int playing=0;    //playing flag

static const u8 *callcmd;    //keeps track of old position when in a call
static int callplaypos;

static u16 defstart[256];  //positions of the defines, relative to songdata

static int uservolume=0x77;  //volume set by user 
        //(combined with Volume command to produce actual volume)
static int userstereo=0xFF;  //stereo flags set by user (combined with Stereo commands)
static int note1,note2,note3,note4;  //current note
static int newnote1,newnote2,newnote3;  //target note (for portamentos)

static u8 wavedata[16];  //buffer for Sound 3 wave fx
static u32 wavecount;  //counter for Sound 3 wave fx
static u32 wavetimer;  //timer for Sound 3 wave fx

static u32 env3timer;  //envelope 3 variables
static u32 env3level;  //current volume of Sound 3 (multiplied by wave data)
static u32 notedelay1,notedelay2,notedelay3,notedelay4;    //Note Delay timers
static u32 noteretrig1,noteretrig2,noteretrig3,noteretrig4;  //Note Retrig timers
static u32 notecut1,notecut2,notecut3,notecut4;      //Note Cut timers

static u32 arpcount1,arpcount2,arpcount3;  //Current Arpeggio states

static u32 vibrdelay1,vibrdelay2,vibrdelay3;  //Vibrato Delay timers
static int vibradd1,vibradd2,vibradd3;    //Vibrato detune values
static int vibrcount1,vibrcount2,vibrcount3;  //Vibrato timers

static u32 pulsesweep1,pulsesweep2;    //Pulse Sweep timers
static u32 volsweep3;        //Volume Sweep 3 timer

static u32 clocksweep4,ratiosweep4,stepswitch4;  //Sound 4 effect timers

//Callback pointers
static DittyNoteCallback *DNC=0;
static DittyFlagCallback *DFC=0;
static DittyEndCallback *DEC=0;

static inline void SetWave(void) {
//Multiplies the current wave data (from wavedata[]) with the cuurent level (env3level)
//and sets the registers.

  int i,j;
  
  if(env3level<=0) {
    REG32(REG_SGWR0)=0;
    REG32(REG_SGWR1)=0;
    REG32(REG_SGWR2)=0;
    REG32(REG_SGWR3)=0;
  } else if(env3level>=15) {
    REG32(REG_SGWR0)=*(long*)(wavedata);
    REG32(REG_SGWR1)=*(long*)(wavedata+4);
    REG32(REG_SGWR2)=*(long*)(wavedata+8);
    REG32(REG_SGWR3)=*(long*)(wavedata+12);
  } else {
    i=15;
    do {
      j=wavedata[i];
      REG8(REG_SGWR0+i)=(((j>>4)*env3level) & 0xF0) |
        (((j & 15)*env3level)>>4);
      i--;
    } while(i>=0);
  }
  REG8(REG_NR30)^=0x40;
}

static void DittyExecute(void) {
  //Executes the current command. For many commands, it simply sets a dittystate[] element.
  int temp;
  switch(currentcmd[1]) {
  case CMD_NOTE1:
    note1=newnote1;
    notedelay1=dittystate[CMD_NOTEDELAY1];
    if(currentcmd[2]!=255) {
      newnote1=(currentcmd[2] & 127)<<3;
      newnote1+=((int)(((u32)dittystate[CMD_TRANSPOSE1])<<25))>>22;
      newnote1+=((int)(((u32)dittystate[CMD_DETUNE1])<<28))>>28;
      if(newnote1<0)
        do { newnote1+=OCTAVE; } while(newnote1<0);
      else if(newnote1>=NOTE_RANGE)
        do { newnote1-=OCTAVE; } while(newnote1>=NOTE_RANGE);
      notecut1=dittystate[CMD_NOTECUT1];
      noteretrig1=dittystate[CMD_NOTERETRIG1];
      arpcount1=(dittystate[CMD_ARPEGGIO1]!=0)?1:0;
      vibrdelay1=dittystate[CMD_VIBRDELAY1];
      vibradd1=dittystate[CMD_VIBRATO1]>>4;
      vibrcount1=0;
      REG8(REG_NR11)=(dittystate[CMD_DUTYCYCLE1]<<6) | dittystate[CMD_SOUNDLEN1];
      pulsesweep1=dittystate[CMD_PULSESWEEP1];
      if(note1==255 || dittystate[CMD_PORTAMENTO1]==0) note1=newnote1;
      if(notedelay1==0) {
        REG8(REG_NR12)=dittystate[CMD_ENVELOPE1];
        REG16(REG_SG11)=0x8000 | NoteVal[note1] | (REG16(REG_SG11) & 0x4000);
      }
    } else {
      note1=newnote1=255;
      if(notedelay1==0) {
        REG8(REG_NR12)=0x00;
        REG16(REG_SG11)=0x8000 | (REG16(REG_SG11) & 0x4000);
      }
    }
    if(DNC!=0) (*DNC)(newnote1,0);
    return;
  case CMD_NOTE2:
    note2=newnote2;
    notedelay2=dittystate[CMD_NOTEDELAY2];
    if(currentcmd[2]!=255) {
      newnote2=(currentcmd[2] & 127)<<3;
      newnote2+=((int)(((u32)dittystate[CMD_TRANSPOSE2])<<25))>>22;
      newnote2+=((int)(((u32)dittystate[CMD_DETUNE2])<<28))>>28;
      if(newnote2<0)
        do { newnote2+=OCTAVE; } while(newnote2<0);
      else if(newnote2>=NOTE_RANGE)
        do { newnote2-=OCTAVE; } while(newnote2>=NOTE_RANGE);
      notecut2=dittystate[CMD_NOTECUT2];
      noteretrig2=dittystate[CMD_NOTERETRIG2];
      arpcount2=(dittystate[CMD_ARPEGGIO2]!=0)?1:0;
      vibrdelay2=dittystate[CMD_VIBRDELAY2];
      vibradd2=dittystate[CMD_VIBRATO2]>>4;
      vibrcount2=0;
      REG8(REG_NR21)=(dittystate[CMD_DUTYCYCLE2]<<6) | dittystate[CMD_SOUNDLEN2];
      pulsesweep2=dittystate[CMD_PULSESWEEP2];
      if(note2==255 || dittystate[CMD_PORTAMENTO2]==0) note2=newnote2;
      if(notedelay2==0) {
        REG8(REG_NR22)=dittystate[CMD_ENVELOPE2];
        REG16(REG_SG21)=0x8000 | NoteVal[note2] | (REG16(REG_SG21) & 0x4000);
      }
    } else {
      note2=newnote2=255;
      if(notedelay2==0) {
        REG8(REG_NR22)=0x00;
        REG16(REG_SG21)=0x8000 | (REG16(REG_SG21) & 0x4000);
      }
    }
    if(DNC!=0) (*DNC)(newnote2,1);
    return;
  case CMD_NOTE3:
    note3=newnote3;
    notedelay3=dittystate[CMD_NOTEDELAY3];
    if(currentcmd[2]!=255) {
      newnote3=(currentcmd[2] & 127)<<3;
      newnote3+=((int)(((u32)dittystate[CMD_TRANSPOSE3])<<25))>>22;
      newnote3+=((int)(((u32)dittystate[CMD_DETUNE3])<<28))>>28;
      if(newnote3<0)
        do { newnote3+=OCTAVE; } while(newnote3<0);
      else if(newnote3>=NOTE_RANGE)
        do { newnote3-=OCTAVE; } while(newnote3>=NOTE_RANGE);
      REG16(REG_SG30_H)=((u16)dittystate[CMD_OUTPUTLVL3]<<13) |
        dittystate[CMD_SOUNDLEN3];
      if(note3==255 || dittystate[CMD_PORTAMENTO3]==0) note3=newnote3;
      notecut3=dittystate[CMD_NOTECUT3];
      noteretrig3=dittystate[CMD_NOTERETRIG3];
      arpcount3=(dittystate[CMD_ARPEGGIO3]!=0)?1:0;
      vibrdelay3=dittystate[CMD_VIBRDELAY3];
      vibradd3=dittystate[CMD_VIBRATO3]>>4;
      vibrcount3=0;
      REG8(REG_NR32)=dittystate[CMD_OUTPUTLVL3]<<5;
      volsweep3=dittystate[CMD_VOLSWEEP3];
      env3level=dittystate[CMD_ENVELOPE3]>>4;
      env3timer=dittystate[CMD_ENVELOPE3] & 7;
      *(u32*)(wavedata)=*(u32*)(dittystate+CMD_WAVEDATA_0);
      *(u32*)(wavedata+4)=*(u32*)(dittystate+CMD_WAVEDATA_4);
      *(u32*)(wavedata+8)=*(u32*)(dittystate+CMD_WAVEDATA_8);
      *(u32*)(wavedata+12)=*(u32*)(dittystate+CMD_WAVEDATA_C);
      wavetimer=dittystate[CMD_WAVESWEEP3]&0xFC;
      wavecount=0;
      SetWave();
      if(notedelay3==0) {
        REG8(REG_NR30)|=0x80;
        REG16(REG_SG31)=0x8000 | NoteVal[note3] | (REG16(REG_SG31) & 0x4000);
      }
    } else {
      note3=newnote3=255;
      if(notedelay3==0) {
        REG8(REG_NR30)=0x00;
        REG16(REG_SG31)=0x8000 | (REG16(REG_SG31) & 0x4000);
      }
    }
    if(DNC!=0) (*DNC)(newnote3,2);
    return;
  case CMD_NOTE4:
    note4=currentcmd[2];
    notedelay4=dittystate[CMD_NOTEDELAY4];
    if(currentcmd[2]==0) {
      REG16(REG_SG40)=((u16)dittystate[CMD_ENVELOPE4]<<8) |
        dittystate[CMD_SOUNDLEN4];
      notecut4=dittystate[CMD_NOTECUT4];
      noteretrig4=dittystate[CMD_NOTERETRIG4];
      REG8(REG_NR43)=dittystate[CMD_NOISEDATA4];
      clocksweep4=dittystate[CMD_CLOCKSWEEP4] & 0x7F;
      ratiosweep4=dittystate[CMD_RATIOSWEEP4] & 0x7F;
      stepswitch4=dittystate[CMD_STEPSWITCH4];
      if(notedelay4==0) {
        REG8(REG_NR42)=dittystate[CMD_ENVELOPE4];
        REG8(REG_NR44)=0x80 | (REG8(REG_NR44) & 0x40);
      }
    } else {
      
      if(notedelay4==0) {
        REG8(REG_NR42)=0x00;
        REG8(REG_NR44)=0x80 | (REG8(REG_NR44) & 0x40);
      }
    }
    if(DNC!=0) (*DNC)(currentcmd[2],3);
    return;
  case CMD_SLUR1:
    if(currentcmd[2]==255) return;
    if(note1==255) note1=newnote1;
    newnote1=(currentcmd[2] & 127)<<3;
    newnote1+=((int)(((u32)dittystate[CMD_TRANSPOSE1])<<25))>>22;
    newnote1+=((int)(((u32)dittystate[CMD_DETUNE1])<<28))>>28;
    if(newnote1<0)
      do { newnote1+=OCTAVE; } while(newnote1<0);
    else if(newnote1>=NOTE_RANGE)
      do { newnote1-=OCTAVE; } while(newnote1>=NOTE_RANGE);
    if(dittystate[CMD_PORTAMENTO1]==0) note1=newnote1;
    return;
  case CMD_SLUR2:
    if(currentcmd[2]==255) return;
    if(note2==255) note2=newnote2;
    newnote2=(currentcmd[2] & 127)<<3;
    newnote2+=((int)(((u32)dittystate[CMD_TRANSPOSE2])<<25))>>22;
    newnote2+=((int)(((u32)dittystate[CMD_DETUNE2])<<28))>>28;
    if(newnote2<0)
      do { newnote2+=OCTAVE; } while(newnote2<0);
    else if(newnote2>=NOTE_RANGE)
      do { newnote2-=OCTAVE; } while(newnote2>=NOTE_RANGE);
    if(dittystate[CMD_PORTAMENTO2]==0) note2=newnote2;
    return;
  case CMD_SLUR3:
    if(currentcmd[2]==255) return;
    if(note3==255) note3=newnote3;
    newnote3=(currentcmd[2] & 127)<<3;
    newnote3+=((int)(((u32)dittystate[CMD_TRANSPOSE3])<<25))>>22;
    newnote3+=((int)(((u32)dittystate[CMD_DETUNE3])<<28))>>28;
    if(newnote3<0)
      do { newnote3+=OCTAVE; } while(newnote3<0);
    else if(newnote3>=NOTE_RANGE)
      do { newnote3-=OCTAVE; } while(newnote3>=NOTE_RANGE);
    if(dittystate[CMD_PORTAMENTO3]==0) note3=newnote3;
    return;
  case CMD_SWEEP1:
    REG8(REG_NR10)=currentcmd[2]; return;
  case CMD_CONTINUOUS1:
    REG16(REG_SG11)=(REG16(REG_SG11) & 0x3FFF) | ((currentcmd[2]!=0)?0:0x4000);
    return;
  case CMD_CONTINUOUS2:
    REG16(REG_SG21)=(REG16(REG_SG21) & 0x3FFF) | ((currentcmd[2]!=0)?0:0x4000);
    return;
  case CMD_CONTINUOUS3:
    REG16(REG_SG31)=(REG16(REG_SG31) & 0x3FFF) | ((currentcmd[2]!=0)?0:0x4000);
    return;
  case CMD_CONTINUOUS4:
    REG16(REG_SG41)=(REG16(REG_SG41) & 0x3FFF) | ((currentcmd[2]!=0)?0:0x4000);
    return;
  case CMD_USERFLAG:
    if(playing && DFC!=0) (*DFC)(currentcmd[2]);
    return;
  case CMD_NEXTSEG:
    if(!playing) return;
    currenttick=currentcmd[2];
    currentplaypos++;
    if(currentplaypos>255 ||
      currentplaypos>songheader->songs[currentsong]) {
        if(DEC==0) {
          DittyStop();
          return;
        }
        else (*DEC)();
    }
    segend=songdata+songheader->segments[songheader->playorder[currentplaypos]];
    currentcmd=songdata;
    if(songheader->playorder[currentplaypos]!=0)
      currentcmd+=songheader->segments[songheader->playorder[currentplaypos]-1];
    if(currentcmd[0]<currenttick && currentcmd<segend)
      do { currentcmd+=3; } while (currentcmd[0]<currenttick && currentcmd<segend);
    currentcmd-=3;
    return;
  case CMD_VOLUME:
    dittystate[CMD_VOLUME]=currentcmd[2];
    temp=uservolume>>4;
    if(temp>0) {
      if(temp<7) {
        temp=((((currentcmd[2] & 0x38)*(temp+1))>>2) & 0x70);
      } else { 
        temp=(currentcmd[2]<<1) & 0x70;
      }
    } else {
      temp=0;
    }
    if((uservolume & 7)>0) {
      if((uservolume & 7)<7) {
        temp|=((currentcmd[2] & 0x07)*((uservolume & 7)+1))>>3;
      } else { 
        temp|=currentcmd[2] & 0x07;
      }
    }
    REG8(REG_NR50)=temp;
    return;
  case CMD_STEREO1:
  case CMD_STEREO2:
  case CMD_STEREO3:
  case CMD_STEREO4:
    dittystate[currentcmd[1]]=currentcmd[2];
    temp=(dittystate[CMD_STEREO4] & 1)|((dittystate[CMD_STEREO4] & 2)<<3);
    temp=(temp<<1)|(dittystate[CMD_STEREO3] & 1)|((dittystate[CMD_STEREO3] & 2)<<3);
    temp=(temp<<1)|(dittystate[CMD_STEREO2] & 1)|((dittystate[CMD_STEREO2] & 2)<<3);
    temp=(temp<<1)|(dittystate[CMD_STEREO1] & 1)|((dittystate[CMD_STEREO1] & 2)<<3);
    REG8(REG_NR51)=temp & userstereo;
    return;
  case CMD_DEFINE:
    if(!playing) return;
    defstart[currentcmd[2]]=currentcmd-songdata;
    return;
  case CMD_ENDDEF:
    if(callcmd==0 || !playing) return;
    currentcmd=callcmd;
    currentplaypos=callplaypos;
    segend=songdata+songheader->segments[songheader->playorder[currentplaypos]];
    currenttick=currentcmd[0];
    callcmd=0;
    return;
  case CMD_CALL:
    if(!playing || defstart[currentcmd[2]]==0xFFFF) return;
    callcmd=currentcmd;
    callplaypos=currentplaypos;
    segend=songdata+0xFFFF; //Fake it, since currentplaypos isn't valid during call.
    currentcmd=songdata+defstart[currentcmd[2]];
    currenttick=currentcmd[0];
    return;
  default:
    if(currentcmd[1]>=DITTY_COMMANDS) return;
    dittystate[currentcmd[1]]=currentcmd[2];
    return;
  }
}

void DittySet(const DittyHeader* const sh,const u8* const sd) {
  //Sets the Ditty header and data.
  songheader=sh; songdata=sd;
}

void DittyPlay(const int song) {
  //Initialize the player to play a song.
  u32 cmd;
  int i;
  REG16(REG_SGCNT0_H)|=0x0002;
  REG16(REG_SGCNT1)=0x0080;
  currentcmd=(u8*)&cmd;
  currentsong=song & 255;
  callcmd=0; playing=0;
  i=0;
  do {
    cmd=((dittydefaults[i]<<8)|i)<<8;
    DittyExecute();
  } while(++i<DITTY_COMMANDS);
  i=0;
  do {
    defstart[i]=0xFFFF;
  } while(++i<256);
  if(currentsong!=0) currentplaypos=songheader->songs[currentsong-1]+1;
  else currentplaypos=0;
  segend=songdata+songheader->segments[songheader->playorder[currentplaypos]];
  currentcmd=songdata;
  if(songheader->playorder[currentplaypos]!=0)
    currentcmd+=songheader->segments[songheader->playorder[currentplaypos]-1];
  currenttick=0;
  currentjiffy=0;
  playing=1;
}

void DittyPlaySegment(const int segment) {
  //Initialize the player to play a segment.
  u32 cmd;
  int i;
  REG16(REG_SGCNT0_H)|=0x0002;
  REG16(REG_SGCNT1)=0x0080;
  currentcmd=(u8*)&cmd;
  currentsong=-1;
  callcmd=0; playing=0;
  i=0;
  do {
    cmd=((dittydefaults[i]<<8)|i)<<8;
    DittyExecute();
  } while(++i<DITTY_COMMANDS);
  i=0;
  do {
    defstart[i]=0xFFFF;
  } while(++i<256);
  currentplaypos=256; // >255 will stop player on end of segment
  segend=songdata+songheader->segments[segment];
  currentcmd=songdata;
  if(segment!=0)
    currentcmd+=songheader->segments[segment-1];
  currenttick=0;
  currentjiffy=0;
  playing=1;
}

void DittyLoop() {
  //Start song over from the beginning. Used as a DittyEndCallback.
  if(currentsong!=0) currentplaypos=songheader->songs[currentsong-1]+1;
  else currentplaypos=0;
  segend=songdata+songheader->segments[songheader->playorder[currentplaypos]];
  currentcmd=songdata;
  if(songheader->playorder[currentplaypos]!=0)
    currentcmd+=songheader->segments[songheader->playorder[currentplaypos]-1];
  if(currentcmd[0]<currenttick && currentcmd<segend)
    do { currentcmd+=3; } while (currentcmd[0]<currenttick && currentcmd<segend);
  currentcmd-=3;
}  

void DittyPlayJiffy() {
  //Plays one jiffy of a song. Must be run every 1/60 of a second (i.e. during VBlank).
  u32 temp;
  if(playing==0) return;
  currentjiffy-=(u32)(dittystate[CMD_JIFFYSTEP])+1;
  if(currentjiffy>0) {
    if(notedelay1>0) {
      if(--notedelay1==0) {
        if(note1!=255) {
          REG8(REG_NR12)=dittystate[CMD_ENVELOPE1];
          REG16(REG_SG11)=0x8000 | NoteVal[note1] | 
            (REG16(REG_SG11) & 0x4000);
        } else {
          REG8(REG_NR12)=0x00;
          REG16(REG_SG11)=0x8000 | (REG16(REG_SG11) & 0x4000);
        }
      }
    } else if(note1!=255) {
      if(note1<newnote1) {
        note1+=dittystate[CMD_PORTAMENTO1];
        if(note1>newnote1) note1=newnote1;
        REG16(REG_SG11)=NoteVal[note1];
      } else if(note1>newnote1) {
        note1-=dittystate[CMD_PORTAMENTO1];
        if(note1<newnote1) note1=newnote1;
        REG16(REG_SG11)=NoteVal[note1];
      }
      temp=note1;
      if(arpcount1!=0) {
        switch(++arpcount1) {
        case 2:
          temp+=(dittystate[CMD_ARPEGGIO1] & 0x0F)<<3; break;
        case 3: 
          if(dittystate[CMD_ARPEGGIO1]>=0x10)
            temp+=(dittystate[CMD_ARPEGGIO1] >>1) & 0x78;
          else arpcount1=1;
          break;
        case 4: arpcount1=1; break;
        }
        if (temp>=NOTE_RANGE) do { temp-=OCTAVE; } while(temp>=NOTE_RANGE);
      }
      if(note1==newnote1) {
        if(vibrdelay1==0) {
          if((dittystate[CMD_VIBRATO1] & 0x0F) !=0) {
            vibrcount1+=vibradd1;
            if(vibrcount1>=(dittystate[CMD_VIBRATO1] & 0x0F)
              || vibrcount1+(dittystate[CMD_VIBRATO1] & 0x0F)
              <=0) {
              vibradd1=-vibradd1;
              vibrcount1=(dittystate[CMD_VIBRATO1] & 0x0F);
              if(vibradd1>0) vibrcount1=-vibrcount1;
              if(temp<0) temp=0;
              else if(temp>=NOTE_RANGE) temp=NOTE_RANGE-1;
            }
            temp+=vibrcount1;
          }
        } else vibrdelay1--;
      }
      REG16(REG_SG11)=NoteVal[temp] | (REG16(REG_SG11) & 0x4000);
      if(pulsesweep1!=0) {
        if(--pulsesweep1==0) {
          pulsesweep1=dittystate[CMD_PULSESWEEP1];
          REG8(REG_NR11)=((REG8(REG_NR11)+0x40) & 0xC0) |
            dittystate[CMD_SOUNDLEN1];
        }
      }
      if(dittystate[CMD_NOTERETRIG1]!=0)
        if(--noteretrig1==0) {
          REG8(REG_NR12)=dittystate[CMD_ENVELOPE1];
          REG16(REG_SG11)=0x8000 | NoteVal[note1] | 
            (REG16(REG_SG11) & 0x4000);
          noteretrig1=dittystate[CMD_NOTERETRIG1];
          notecut1=dittystate[CMD_NOTECUT1];
        }
      if(notecut1!=0)
        if(--notecut1==0) {
          REG8(REG_NR12)=0x00;
          REG16(REG_SG11)=0x8000 | (REG16(REG_SG11) & 0x4000);
        }
    }
    if(notedelay2>0) {
      if(--notedelay2==0) {
        if(note2!=255) {
          REG8(REG_NR22)=dittystate[CMD_ENVELOPE2];
          REG16(REG_SG21)=0x8000 | NoteVal[note2] | 
            (REG16(REG_SG21) & 0x4000);
        } else {
          REG8(REG_NR22)=0x00;
          REG16(REG_SG21)=0x8000 | (REG16(REG_SG21) & 0x4000);
        }
      }
    } else if(note2!=255) {
      if(note2<newnote2) {
        note2+=dittystate[CMD_PORTAMENTO2];
        if(note2>newnote2) note2=newnote2;
        REG16(REG_SG21)=NoteVal[note2];
      } else if(note2>newnote2) {
        note2-=dittystate[CMD_PORTAMENTO2];
        if(note2<newnote2) note2=newnote2;
        REG16(REG_SG21)=NoteVal[note2];
      }
      temp=note2;
      if(arpcount2!=0) {
        switch(++arpcount2) {
        case 2:
          temp+=(dittystate[CMD_ARPEGGIO2] & 0x0F)<<3; break;
        case 3: 
          if(dittystate[CMD_ARPEGGIO2]>=0x10)
            temp+=(dittystate[CMD_ARPEGGIO2] >>1) & 0x78;
          else arpcount2=1;
          break;
        case 4: arpcount2=1; break;
        }
        if (temp>=NOTE_RANGE) do { temp-=OCTAVE; } while(temp>=NOTE_RANGE);
      }
      if(note2==newnote2) {
        if(vibrdelay2==0) {
          if((dittystate[CMD_VIBRATO2] & 0x0F) !=0) {
            vibrcount2+=vibradd2;
            if(vibrcount2>=(dittystate[CMD_VIBRATO2] & 0x0F)
              || vibrcount2+(dittystate[CMD_VIBRATO2] & 0x0F)
              <=0) {
              vibradd2=-vibradd2;
              vibrcount2=(dittystate[CMD_VIBRATO2] & 0x0F);
              if(vibradd2>0) vibrcount2=-vibrcount2;
              if(temp<0) temp=0;
              else if(temp>=NOTE_RANGE) temp=NOTE_RANGE-1;
            }
            temp+=vibrcount2;
          }
        } else vibrdelay2--;
      }
      REG16(REG_SG21)=NoteVal[temp] | (REG16(REG_SG21) & 0x4000);
      if(pulsesweep2!=0) {
        if(--pulsesweep2==0) {
          pulsesweep2=dittystate[CMD_PULSESWEEP2];
          REG8(REG_NR21)=((REG8(REG_NR21)+0x40) & 0xC0) |
            dittystate[CMD_SOUNDLEN2];
        }
      }
      if(dittystate[CMD_NOTERETRIG2]!=0)
        if(--noteretrig2==0) {
          REG8(REG_NR22)=dittystate[CMD_ENVELOPE2];
          REG16(REG_SG21)=0x8000 | NoteVal[note2] | 
            (REG16(REG_SG21) & 0x4000);
          noteretrig2=dittystate[CMD_NOTERETRIG2];
          notecut2=dittystate[CMD_NOTECUT2];
        }
      if(notecut2!=0)
        if(--notecut2==0) {
          REG8(REG_NR22)=0x00;
          REG16(REG_SG21)=0x8000 | (REG16(REG_SG21) & 0x4000);
        }
    }
    if(notedelay3>0) {
      if(--notedelay3==0) {
        if(note3!=255) {
          REG8(REG_NR30)|=0x80;
          REG16(REG_SG31)=0x8000 | NoteVal[note3] | 
            (REG16(REG_SG31) & 0x4000);
          env3level=(dittystate[CMD_ENVELOPE3]>>4);
          env3timer=dittystate[CMD_ENVELOPE3] & 7;
          SetWave();
        } else {
          REG8(REG_NR30)=0x00;
          REG16(REG_SG31)=0x8000 | (REG16(REG_SG31) & 0x4000);
        }
      }
    } else if(note3!=255) {
      if(note3<newnote3) {
        note3+=dittystate[CMD_PORTAMENTO3];
        if(note3>newnote3) note3=newnote3;
        REG16(REG_SG31)=NoteVal[note3];
      } else if(note3>newnote3) {
        note3-=dittystate[CMD_PORTAMENTO3];
        if(note3<newnote3) note3=newnote3;
        REG16(REG_SG31)=NoteVal[note3];
      }
      temp=note3;
      if(arpcount3!=0) {
        switch(++arpcount3) {
        case 2:
          temp+=(dittystate[CMD_ARPEGGIO3] & 0x0F)<<3; break;
        case 3: 
          if(dittystate[CMD_ARPEGGIO3]>=0x10)
            temp+=(dittystate[CMD_ARPEGGIO3] >>1) & 0x78;
          else arpcount3=1;
          break;
        case 4: arpcount3=1; break;
        }
        if (temp>=NOTE_RANGE) do { temp-=OCTAVE; } while(temp>=NOTE_RANGE);
      }
      if(note3==newnote3) {
        if(vibrdelay3==0) {
          if((dittystate[CMD_VIBRATO3] & 0x0F) !=0) {
            vibrcount3+=vibradd3;
            if(vibrcount3>=(dittystate[CMD_VIBRATO3] & 0x0F)
              || vibrcount3+(dittystate[CMD_VIBRATO3] & 0x0F)
              <=0) {
              vibradd3=-vibradd3;
              vibrcount3=(dittystate[CMD_VIBRATO3] & 0x0F);
              if(vibradd3>0) vibrcount3=-vibrcount3;
              if(temp<0) temp=0;
              else if(temp>=NOTE_RANGE) temp=NOTE_RANGE-1;
            }
            temp+=vibrcount3;
          }
        } else vibrdelay3--;
      }
      REG16(REG_SG31)=NoteVal[temp] | (REG16(REG_SG31) & 0x4000);
      if(volsweep3!=0) {
        if(--volsweep3==0) {
          static const u8 volsweep3up[8]={0x60,0x00,0x80,0x40,
            0x20,0x20,0x20,0x20};
          static const u8 volsweep3dn[8]={0x20,0x80,0x60,0x00,
            0x40,0x40,0x40,0x40};
          volsweep3=dittystate[CMD_VOLSWEEP3] & 127;
          if((dittystate[CMD_VOLSWEEP3] & 128)==0)
            REG8(REG_NR32)=volsweep3up[REG8(REG_NR32)>>5];
          else REG8(REG_NR32)=volsweep3dn[REG8(REG_NR32)>>5];
        }
      }
      if(dittystate[CMD_NOTERETRIG3]!=0)
        if(--noteretrig3==0) {
          REG8(REG_NR30)=0x80;
          REG16(REG_SG31)=0x8000 | NoteVal[note3] | 
            (REG16(REG_SG31) & 0x4000);
          noteretrig3=dittystate[CMD_NOTERETRIG3];
          notecut3=dittystate[CMD_NOTECUT3];
          env3level=(dittystate[CMD_ENVELOPE3]>>4);
          env3timer=dittystate[CMD_ENVELOPE3] & 7;
          SetWave();
        }
      if(notecut3!=0)
        if(--notecut3==0) {
          REG8(REG_NR30)=0x00;
          REG16(REG_SG31)=0x8000 | (REG16(REG_SG31) & 0x4000);
        }
      temp=0;
      if(env3timer!=0) {
        if(--env3timer==0) {
          if((dittystate[CMD_ENVELOPE3] & 0x08)==0) {
            if(env3level>0) {
              env3level--;
              env3timer=dittystate[CMD_ENVELOPE3] & 7;
              temp=1;
            }
          } else {
            if(env3level<15) {
              env3level++;
              env3timer=dittystate[CMD_ENVELOPE3] & 7;
              temp=1;
            }
          }
        }
      }
      if(wavetimer!=0) {
        if(--wavetimer==0) {
          switch (dittystate[CMD_WAVESWEEP3]&3) {
          case 3:
            wavedata[4^((wavecount>>1)&15)]^=(wavecount&1)?0x0F:0xF0;
            wavedata[12^((wavecount>>1)&15)]^=(wavecount&1)?0x0F:0xF0;
          case 2:
            wavedata[8^((wavecount>>1)&15)]^=(wavecount&1)?0x0F:0xF0;
          case 1:
            wavedata[(wavecount>>1)&15]^=(wavecount&1)?0x0F:0xF0;
            temp=1;
            wavetimer=dittystate[CMD_WAVESWEEP3]&0xFC;
            wavecount++;
          }
        }
      }
      if(temp) SetWave();
    }
    if(notedelay4>0) {
      if(--notedelay4==0) {
        if(note4!=255) {
          REG8(REG_NR42)=dittystate[CMD_ENVELOPE4];
          REG8(REG_NR44)=0x80 | (REG8(REG_NR44) & 0x40);
        } else {
          REG8(REG_NR42)=0x00;
          REG8(REG_NR44)=0x80 | (REG8(REG_NR44) & 0x40);
        }
      }
    } else if(note4==0) {
      temp=REG8(REG_NR43);
      if(clocksweep4!=0) {
        if(--clocksweep4==0) {
          clocksweep4=dittystate[CMD_CLOCKSWEEP4] & 0x7F;
          if((dittystate[CMD_CLOCKSWEEP4]&0x80)==0) {
            temp+=0x10;
            if(temp>=0xE0) temp&=0x0F;
          } else {
            if(temp<0x10) temp|=0xE0;
            temp-=0x10;
          }
        }
      }
      if(ratiosweep4!=0) {
        if(--ratiosweep4==0) {
          ratiosweep4=dittystate[CMD_RATIOSWEEP4] & 0x7F;
          if((dittystate[CMD_RATIOSWEEP4]&0x80)==0) 
            temp=(temp & 0xF8) | ((temp+0x01) & 0x07);
          else temp=(temp & 0xF8) | ((temp-0x01) & 0x07);
        }
      }
      if(stepswitch4!=0) {
        if(--stepswitch4==0) {
          stepswitch4=dittystate[CMD_STEPSWITCH4];
          temp^=0x08;
        }
      }
      REG8(REG_NR43)=temp;
      if(dittystate[CMD_NOTERETRIG4]!=0)
        if(--noteretrig4==0) {
          REG8(REG_NR42)=dittystate[CMD_ENVELOPE4];
          REG8(REG_NR44)=0x80 | (REG8(REG_NR44) & 0x40);
          noteretrig4=dittystate[CMD_NOTERETRIG4];
          notecut4=dittystate[CMD_NOTECUT4];
        }
      if(notecut4!=0)
        if(--notecut4==0) {
          REG8(REG_NR42)=0x00;
          REG8(REG_NR44)=0x80 | (REG8(REG_NR44) & 0x40);
        }
    }
    return;
  }
  if(currentcmd[0]<=currenttick && currentcmd<segend)
    do {
      DittyExecute();
      currentcmd+=3;
    } while (currentcmd[0]<=currenttick && currentcmd<segend);
  currentjiffy += dittystate[CMD_TEMPO] << 8;
  currenttick++;
  if(currenttick>=256) {
    currenttick=0;
    currentplaypos++;
    if(currentplaypos>255 || 
      currentplaypos>songheader->songs[currentsong]) {
        if(DEC==0) {
          DittyStop();
          return;
        }
        else (*DEC)();
    }
    segend=songdata+songheader->segments[songheader->playorder[currentplaypos]];
    currentcmd=songdata;
    if(songheader->playorder[currentplaypos]!=0)
      currentcmd+=songheader->segments[songheader->playorder[currentplaypos]-1];
  }
}

void DittyStop() {
//End Ditty Playback
  REG16(REG_SG10_H)=0x0000;
  REG16(REG_SG11)=0x8000;
  REG16(REG_SG20)=0x0000;
  REG16(REG_SG21)=0x8000;
  REG16(REG_SG30_H)=0x0000;
  REG16(REG_SG31)=0x8000;
  REG16(REG_SG40)=0x0000;
  REG16(REG_SG41)=0x8000;
  playing=0;
}

int DittyGetTick() {
  return currenttick;
}

int DittyGetSegment() {
  return songheader->playorder[currentplaypos];
}

int DittyGetPlayPos() {
  return currentplaypos;
}

void DittySetNoteCallback(DittyNoteCallback *D) {
  DNC=D;
}

void DittySetFlagCallback(DittyFlagCallback *D) {
  DFC=D;
}

void DittySetEndCallback(DittyEndCallback *D) {
  DEC=D;
}

void DittySetVolume(const int Left, const int Right) {
  int v;
  if(Left>0) {
    if(Left<7) {
      uservolume=Left<<3;
      v=((((dittystate[CMD_VOLUME] & 0x38)*(Left+1))>>2) & 0x70);
    } else { 
      uservolume=0x38;
      v=dittystate[CMD_VOLUME] & 0x70;
    }
  } else {
    uservolume=0;
    v=0;
  }
  if(Right>0) {
    if(Right<7) {
      uservolume|=Right;
      v|=((dittystate[CMD_VOLUME] & 0x07)*(Right+1))>>3;
    } else { 
      uservolume|=0x07;
      v|=dittystate[CMD_VOLUME] & 0x07;
    }
  }
  REG8(REG_NR50)=v;
}

void DittySetStereo(const int Flags) {
  int s;
  userstereo=Flags;
  s=(dittystate[CMD_STEREO4] & 1)|((dittystate[CMD_STEREO4] & 2)<<3);
  s=(s<<1)|(dittystate[CMD_STEREO3] & 1)|((dittystate[CMD_STEREO3] & 2)<<3);
  s=(s<<1)|(dittystate[CMD_STEREO2] & 1)|((dittystate[CMD_STEREO2] & 2)<<3);
  s=(s<<1)|(dittystate[CMD_STEREO1] & 1)|((dittystate[CMD_STEREO1] & 2)<<3);
  REG8(REG_NR51)=s & userstereo;
}

int DittyPlaying(void) {
  return playing;
}
